<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\Trainer;
use App\Models\User;
use App\Models\UserHierarchy;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Devrabiul\ToastMagic\Facades\ToastMagic;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $data['title'] = "Students";
        $user = $request->user();
        if($user->role == 'manager'){
            $request->merge(['role' => 'student']);
        }

        $query = User::query();

        if ($request->role) {
            $query->where('role', $request->role);
        }
        if($request->status == 'pending'){
            $query->whereNull('verified_at');
        }

        if ($request->search) {
            $query->where('id', $request->search)
                ->orWhere('name', 'like', '%' . $request->search . '%')
                ->orWhere('phone', 'like', '%' . $request->search . '%');
        }

        $data['users'] = $query->latest()->paginate(25)->appends($request->query());
        $data['roles'] = ['admin', 'manager', 'student', 'counselor', 'stl', 'tl', 'trainer', 'teacher'];
        if($user->role == 'manager'){
            return view('manager.students', $data);
        }
        return view('backend.users.index', $data);
    }

    public function search(Request $request){
        $user = $request->user();

        $data['title'] = "Search Students";

        if(!$request->search && !$request->from && !$request->to){
            $data['users'] = new LengthAwarePaginator([], 0, 25);
        }
        else{
            $query = User::query()->with('referer:id,name')->where('role', 'student');

            if ($request->search) {
                $query->where('id', $request->search)
                    ->orWhere('phone', 'like', '%' . $request->search . '%');
            }else{
                if($request->status == 'pending'){
                    $query->whereNull('verified_at');
                }else{
                    $query->whereNotNull('verified_at');
                }
                $query->whereBetween('created_at', [Carbon::parse($request->from)->startOfDay(), Carbon::parse($request->to)->endOfDay()]);
            }

            $data['users'] = $query->latest()->paginate(25)->appends($request->query());
        }
        if($user->role == 'manager'){
            return view('manager.search_student', $data);
        }
        return view('backend.users.search', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $data['title'] = "Add new Student";
        $user = $request->user();
        if($user->role == 'manager'){
            $request->merge(['role' => 'student']);
        }

        $request->validate([
            'role' => 'nullable|in:admin,manager,student,counselor,stl,tl,trainer,teacher'
        ]);
        $role = $request->input('role', 'student');

        if (in_array($role, ['stl', 'tl', 'trainer', 'teacher', 'student'])) {
            $data['counselors'] = User::where('role', 'counselor')->latest()->get(['id', 'name']);
        }
        if (in_array($role, ['trainer', 'teacher'])) {
            $data['courses'] = Course::latest()->get(['id', 'name']);
        }
        $data['role'] = $role;
        if($user->role == 'manager'){
            return view('manager.add_student', $data);
        }
        return view('backend.users.add', $data);
    }

    /**
     * Store a newly create resource in storage.
     */
    public function store(Request $request)
    {
        $user = $request->user();
        if($user->role == 'manager'){
            $request->merge(['role' => 'student']);
        }

        $baseRules = [
            'name' => ['required', 'string', 'max:255'],
            'role' => ['required', 'in:admin,manager,student,counselor,stl,tl,trainer,teacher'],
            'phone' => ['required', 'unique:users,phone'],
            'password' => ['required', 'string'],
        ];

        $roleSpecificRules = match ($request->role) {
            'stl' => ['counselor' => 'required'],
            'tl' => ['stl' => 'required'],
            'trainer' => ['tl' => 'required'],
            'teacher' => ['tl' => 'required'],
            'student' => ['tl' => 'required'],
            default => [],
        };

        $request->validate(array_merge($baseRules, $roleSpecificRules));


        DB::transaction(function () use ($request) {
            $role = $request->role;

            $user = User::create([
                'name' => $request->name,
                'role' => $role,
                'phone' => $request->phone,
                'email' => $request->email,
                'gender' => $request->gender,
                'address' => $request->address,
                'payment_status' => $request->payment_status,
                'status' => $request->status,
                'password' => bcrypt($request->password),
                'verified_at' => now()
            ]);

            if ($role == 'stl') {
                UserHierarchy::create([
                    'user_id' => $user->id,
                    'counselor_id' => $request->counselor,
                    'assigned_by' => $request->user()->id
                ]);
            } else if ($role == 'tl') {
                UserHierarchy::create([
                    'user_id' => $user->id,
                    'senior_team_leader_id' => $request->stl,
                    'assigned_by' => $request->user()->id
                ]);
            } else if ($role == 'trainer') {
                Trainer::create([
                    'user_id' => $user->id,
                    'course_id' => $request->course,
                    'team_leader_id' => $request->tl
                ]);
            } else if ($role == 'teacher') {
                Teacher::create([
                    'user_id' => $user->id,
                    'course_id' => $request->course,
                    'team_leader_id' => $request->tl
                ]);
            } else if ($role == 'student') {
                Student::create([
                    'user_id' => $user->id,
                    'team_leader_id' => $request->tl
                ]);
            }
        });

        ToastMagic::success("New user has been create");
        return redirect()->back();
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $auth = Auth::user();
        if($auth->role == 'manager' && $user->role != 'student'){
            abort(404);
        }
        $data['user'] = $user->load(['student.trainer', 'student.teacher']);
        $data['role'] = $user->role;
        if($auth->role == 'manager'){
            $data['title'] = "Student Profile";
            return view('manager.student-details', $data);
        }
        return view('backend.users.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $auth = Auth::user();
        if($auth->role == 'manager' && $user->role != 'student'){
            abort(404);
        }

        $data['user'] = $user;
        if (in_array($user->role, ['stl', 'tl', 'trainer', 'teacher', 'student'])) {
            $data['counselors'] = User::where('role', 'counselor')->latest()->get(['id', 'name']);
        }
        if (in_array($user->role, ['trainer', 'teacher'])) {
            $data['courses'] = Course::latest()->get(['id', 'name']);
        }
        $data['role'] = $user->role;
        if($auth->role == 'manager'){
            return view('manager.edit_student', $data);
        }
        return view('backend.users.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $auth = Auth::user();
        if($auth->role == 'manager' && $user->role != 'student'){
            abort(404);
        }

        $baseRules = [
            'name' => ['required', 'string', 'max:255'],
            'phone' => ['required', 'unique:users,phone,' . $user->id]
        ];

        $roleSpecificRules = match ($request->role) {
            'stl' => ['counselor' => 'required'],
            'tl' => ['stl' => 'required'],
            'trainer' => ['tl' => 'required'],
            'teacher' => ['tl' => 'required'],
            'student' => ['tl' => 'required'],
            default => [],
        };

        $request->validate(array_merge($baseRules, $roleSpecificRules));

        $role = $user->role;
        DB::transaction(function () use ($request, $user, $role) {

            $user->update([
                'name' => $request->name,
                'phone' => $request->phone,
                'email' => $request->email,
                'gender' => $request->gender,
                'address' => $request->address,
                'payment_status' => $request->payment_status,
                'status' => $request->status,
                'password' => $request->password ? bcrypt($request->password) : $user->password,
            ]);

            if ($role == 'stl') {
                UserHierarchy::updateOrCreate(
                    ['user_id' => $user->id],
                    [
                        'counselor_id' => $request->counselor,
                        'assigned_by' => $request->user()->id,
                    ]
                );
            } else if ($role == 'tl') {
                UserHierarchy::updateOrCreate(
                    ['user_id' => $user->id],
                    [
                        'senior_team_leader_id' => $request->stl,
                        'assigned_by' => $request->user()->id,
                    ]
                );
            } else if ($role == 'trainer') {
                $tl = $user->teamLeaderOfTrainer;
                if ($tl->id != $request->tl) {
                    Student::where('trainer_id', $user->id)->update([
                        'trainer_id' => null
                    ]);
                }
                $user->trainer()->updateOrCreate(
                    ['user_id' => $user->id],
                    [
                        'course_id' => $request->course,
                        'team_leader_id' => $request->tl
                    ]
                );
            } else if ($role == 'teacher') {
                $tl = $user->teamLeaderOfTeacher;
                if ($tl->id != $request->tl) {
                    Student::where('teacher_id', $user->id)->update([
                        'teacher_id' => null
                    ]);
                }
                $user->teacher()->updateOrCreate(
                    ['user_id' => $user->id],
                    [
                        'course_id' => $request->course,
                        'team_leader_id' => $request->tl
                    ]
                );
            } else if ($role == 'student') {
                $user->student()->updateOrCreate(
                    ['user_id' => $user->id],
                    [
                        'team_leader_id' => $request->tl
                    ]
                );

                if(!$user->verified_at){
                    //calculate commissions
                    $user->verified_at = now();
                    $user->save();

                    $referer = $user->referer;
                    if(!$referer){
                        return;
                    }

                    $settings = config('app.settings');
                    $details = "Student(#$user->id) activation commission";

                    $tl = $referer->teamLeaderOfStudent;
                    $stl = $tl?->seniorTeamLeader;
                    $cs = $stl?->counselor;
                    $tr_tc = $referer->student;
                    $manager = User::where('role', 'manager')->first();
                    
                    if($amount = $settings['student_activation_cashback']){
                        credit_balance($user->id, $amount, "Activation cashback");
                    }
                    if($amount = $settings['student_activation_commission_to_referrer']){
                        credit_balance($referer->id, $amount, $details);
                    }
                    if($tl && $amount = $settings['student_activation_commission_to_tl']){
                        credit_balance($tl->id, $amount, $details);
                    }
                    if($stl && $amount = $settings['student_activation_commission_to_stl']){
                        credit_balance($stl->id, $amount, $details);
                    }
                    if($cs && $amount = $settings['student_activation_commission_to_counselor']){
                        credit_balance($cs->id, $amount, $details);
                    }
                    if($tr_tc && $tr_tc?->trainer_id && $amount = $settings['student_activation_commission_to_tr']){
                        credit_balance($tr_tc->trainer_id, $amount, $details);
                    }
                    if($tr_tc && $tr_tc?->teacher_id && $amount = $settings['student_activation_commission_to_teacher']){
                        credit_balance($tr_tc->teacher_id, $amount, $details);
                    }
                    if($manager && $amount = $settings['student_activation_commission_to_manager']){
                        credit_balance($manager->id, $amount, $details);
                    }
                }
            }
        });

        ToastMagic::success("User has been updated");
        if($request->source){
            return redirect($request->source);
        }
        return redirect()->route('users.index', ['role' => $role]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        $auth = Auth::user();
        if($auth->role == 'manager' && $user->role != 'student'){
            abort(404);
        }
        DB::transaction(function() use($user){
            $user->transactions()->delete();
            $user->delete();
        });
        ToastMagic::success("User has been deleted");
        return redirect()->back();
    }
}