<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Payment\UddoktaPay;
use App\Models\Course;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Withdrawal;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Devrabiul\ToastMagic\Facades\ToastMagic;
use Illuminate\Support\Facades\DB;

class HomeController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth')->except('home');
    }
    public function home(){
        $data['courses'] = Course::latest()->get();
        $data['user'] = Auth::user()?->load(['trainer', 'teacher', 'teamLeaderOfStudent.seniorTeamLeader.counselor']);
        return view('welcome', $data);
    }

    public function index()
    {
        $data['title'] = "Dashboard";
        $data['user'] = Auth::user();
        $data['totalWithdraws'] = Withdrawal::where([['user_id', $data['user']->id], ['status', 'approved']])->sum('amount') ?? 0;
        $data['totalLeads'] = User::where('refer_by', $data['user']->id)->count() ?? 0;
        if ($data['user']->role == 'admin') {
            return redirect()->route('admin.dashboard');
        }
        return view('dashboard', $data);
    }

    public function showProfile()
    {
        $data['user'] = Auth::user();
        $data['title'] = "Profile";
        return view('profile', $data);
    }
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        $request->validate([
            'name' => 'required|max:255',
            'email' => 'nullable|email',
            'phone' => 'required|unique:users,phone,' . $user->id,
            'avatar' => 'nullable|image|max:3000'
        ]);

        $user->update([
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'gender' => $request->gender,
            'address' => $request->address,
            'password' => $request->password ? bcrypt($request->password) : $user->password,
            'avatar' => updateImg($request->file('avatar'), 'avatar', $user->avatar),
        ]);

        ToastMagic::success('Profile has been updated');

        return redirect()->back();
    }

    public function referralPage()
    {
        $data['user'] = Auth::user();
        $data['title'] = "My Referrals";

        $userId = $data['user']->id;
        $today = Carbon::today();
        $tomorrow = Carbon::tomorrow(); // for range upper bound

        // Get the paginated leads
        $data['leads'] = User::where('refer_by', $userId)
            ->latest()
            ->paginate(25);

        $stats = User::selectRaw(
            "
            COUNT(CASE WHEN verified_at IS NULL THEN 1 END) as totalLeads,
            COUNT(CASE WHEN verified_at IS NULL AND created_at BETWEEN ? AND ? THEN 1 END) as todayLeads,
            COUNT(CASE WHEN verified_at IS NOT NULL THEN 1 END) as totalConverted,
            COUNT(CASE WHEN verified_at IS NOT NULL AND created_at BETWEEN ? AND ? THEN 1 END) as todayConverted",
            [$today, $tomorrow, $today, $tomorrow]
        )
            ->where('refer_by', $userId)
            ->first();
        $data['totalLeads'] = $stats->totalLeads;
        $data['todayLeads'] = $stats->todayLeads;
        $data['totalConverted'] = $stats->totalConverted;
        $data['todayConverted'] = $stats->todayConverted;
        return view('referral', $data);
    }

    public function passbook()
    {
        $data['title'] = "My Passbook";
        $data['transactions'] = Transaction::where('user_id', Auth::id())->latest()->paginate(25);
        return view('passbook', $data);
    }

    public function withdrawal()
    {
        $data['title'] = "My Withdrawals";
        $data['user'] = Auth::user();
        $data['withdrawals'] = Withdrawal::where('user_id', $data['user']->id)->latest()->paginate(25);
        return view('withdrawal', $data);
    }
    public function storeWithdrawal(Request $request)
    {
        $user = Auth::user();

        try {
            DB::transaction(function () use ($request, $user) {
                $settings = config('app.settings');
                if (!$user->pending_fee) {
                    debit_balance($user->id, $settings['pending_fee'], 'Pending fee paid');
                    $user->pending_fee = now();
                    $user->save();

                    ToastMagic::success("Thank you, Pending fee has been paid.");
                    return;
                }

                debit_balance($user->id, $request->amount, 'Balance withdrawal');
                $details = [
                    'account_number' => $request->account_number,
                    'method' => $request->method,
                    'type' => $request->type
                ];

                Withdrawal::create([
                    'user_id' => $user->id,
                    'amount' => $request->amount,
                    'details' => json_encode($details)
                ]);

                $user->last_withdraw = now();
                $user->save();
                ToastMagic::success("Withdrawal request has been sent");
            });
        } catch (\Throwable $th) {
            ToastMagic::error($th->getMessage());
        }

        return redirect()->back();
    }

    public function activationView()
    {
        $data['title'] = 'Activate Account';
        $data['user'] = Auth::user();

        if ($data['user']->verified_at) {
            return redirect()->route('dashboard');
        }

        return view('activation', $data);
    }

    public function activationPay(Request $request)
    {
        $settings = config('app.settings');

        if (!$settings['payment_gateway'] || !$settings['activation_fee']) {
            return redirect()->back();
        }

        $user = $request->user();
        if ($user->payment_status == 'paid') {
            return redirect()->back();
        }

        $requestData = [
            'full_name'    => $user->name,
            'email'        => $user->email ?? $settings['email'],
            'amount'       => $settings['activation_fee'],
            'metadata'     => [
                'type'   => 'activation_fee',
                'user_id' => $user->id,
            ],
            'redirect_url' => route('payment.success'),
            'return_type' => 'GET',
            'cancel_url'   => route('payment.cancel'),
        ];

        try {
            return redirect()->away((new UddoktaPay())->init_payment($requestData));
        } catch (\Exception $exception) {
            return $exception->getMessage();
        }
    }

    public function successPayment(Request $request)
    {
        $request->validate([
            'invoice_id' => 'required'
        ]);

        $user = Auth::user();

        $invoiceId = $request->invoice_id;
        $response = (new UddoktaPay())->verify_payment(['invoice_id' => $invoiceId]);

        if ($response['status'] == 'ERROR') {
            ToastMagic::error('Payment failed!');
            return redirect('/activation');
        }

        $userId = $response['metadata']['user_id'];
        if ($userId != $user->id) {
            ToastMagic::error('Authentication failed!');
            return redirect('/activation');
        }

        if ($response['status'] == 'COMPLETED') {
            $user->update([
                'payment_status' => 'paid'
            ]);
            ToastMagic::success('Payment completed!');
        } else {
            ToastMagic::warning("Payment status: " . $response['status'] . '| Please contact admin');
        }
        return redirect('/activation');
    }
    public function cancelPayment()
    {
        ToastMagic::error('Authentication cancelled!');
        return redirect('/activation');
    }
}
